' SAP Cleaning Check Script
' Author: Dario Pascoal
'
' Description: This script determines if a tool requires special EUV cleaning procedures
' by analyzing its production hierarchy in SAP. The script connects to SAP, searches for
' a specific material number using the ZSTS transaction, and examines the production
' hierarchy field to identify tools that contain SPEC EUV, COM EUV, or TBA indicators.
'
' Tools with these indicators require specialized cleaning procedures due to their use
' in EUV (Extreme Ultraviolet) lithography processes or other sensitive applications.
' The script provides a clear yes/no answer about cleaning requirements to support
' operational decision-making in the DHL tool management workflow.
'
' Prerequisites: Active SAP GUI session with access to ZSTS transaction
' Parameters:
'   - Argument 1: Material Number (required) - The 12NC material number to check
'   - Argument 2: Log file path (optional) - Path for detailed logging output
' Returns: Structured output indicating EUV cleaning requirements and hierarchy details
'
On Error Resume Next

' Configure SAP GUI automation environment
' This enables programmatic control of the SAP interface
Set WSHShell = CreateObject("WScript.Shell")

' Disable SAP GUI scripting security warnings in Windows registry
' These settings prevent popup dialogs that would interrupt the automation process
' Each setting controls a different aspect of SAP GUI security warnings
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAttach", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnConnection", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAllowListRequired", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\UseAllowList", 0, "REG_DWORD"

' Extract material number from command line arguments
' This is the primary identifier (12NC) for the tool we need to analyze
materialNumber = WScript.Arguments.Item(0)

' Initialize logging system for debugging and audit purposes
' Variables to handle file-based logging throughout the script execution
Dim logFile, logFilePath

' Set up optional logging if log file path is provided as second argument
' This allows detailed tracking of script execution for troubleshooting
If WScript.Arguments.Count > 1 Then
    logFilePath = WScript.Arguments.Item(1)
    ' Attempt to open log file in append mode (mode 8) to preserve existing entries
    On Error Resume Next
    Set fso = CreateObject("Scripting.FileSystemObject")
    Set logFile = fso.OpenTextFile(logFilePath, 8, True)
    
    ' Handle potential file access errors gracefully
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not open log file: " & Err.Description
        Err.Clear
        Set logFile = Nothing
    Else
        ' Create clear log entry header for this script execution
        LogToFile "========================================="
        LogToFile "Cleaning Check Script Started at " & Now
        LogToFile "Script: " & WScript.ScriptName
        LogToFile "Material Number: " & materialNumber
        LogToFile "========================================="
    End If
End If

' Helper function to write messages to log file only
' This supports detailed debugging without cluttering console output
' Parameters: message - The text to write to the log file with timestamp
Sub LogToFile(message)
    On Error Resume Next
    If Not logFile Is Nothing Then
        ' Write formatted log entry with timestamp and script identification
        logFile.WriteLine Now & " - [" & WScript.ScriptName & "] - " & message
    End If
End Sub

' Helper function to output messages to both console and log file
' This ensures important information is visible immediately and permanently recorded
' Parameters: message - The text to display and log
Sub LogMessage(message)
    WScript.Echo message    ' Display immediately for user feedback
    LogToFile message       ' Preserve in log file for audit trail
End Sub

' Establish connection to existing SAP GUI session
' This script requires an active SAP session to be already running
' The connection hierarchy follows: Application -> Connection -> Session
If Not IsObject(application) Then
   Set SapGuiAuto = GetObject("SAPGUI")
   If Err.Number <> 0 Then
       LogMessage "Error getting SAPGUI: " & Err.Description
       WScript.Echo "ERROR: Cannot connect to SAP - " & Err.Description
       WScript.Quit 1
   End If
   Set application = SapGuiAuto.GetScriptingEngine
End If

If Not IsObject(connection) Then
   Set connection = application.Children(0)
End If

If Not IsObject(session) Then
   Set session = connection.Children(0)
End If

' Enable event handling for SAP GUI automation
If IsObject(WScript) Then
   WScript.ConnectObject session, "on"
   WScript.ConnectObject application, "on"
End If

' Handle potential SAP GUI security dialog automatically
' This function manages security warnings that may appear during automation
' It automatically accepts the security dialog to prevent script interruption
Sub HandleSecurityDialog()
    Dim shell, timeout, dialogFound
    Set shell = CreateObject("WScript.Shell")
    timeout = 0
    dialogFound = False
    
    ' Check for security dialog for up to 10 seconds
    Do While timeout < 10
        ' Look for the SAP GUI Security dialog window
        If shell.AppActivate("SAP GUI Security") Then
            WScript.Sleep 500
            shell.SendKeys " "  ' Press space to check "Remember my decision" checkbox
            WScript.Sleep 200
            shell.SendKeys "{ENTER}"  ' Press Enter to click Allow button
            dialogFound = True
            Exit Do
        End If
        WScript.Sleep 500
        timeout = timeout + 1
    Loop
End Sub

' Execute the security dialog handler
HandleSecurityDialog()

' Begin the production hierarchy analysis process
' This section navigates to the ZSTS transaction and configures the search parameters
On Error Resume Next
LogMessage "Checking production hierarchy for material: " & materialNumber

' Navigate to ZSTS transaction for production hierarchy analysis
' ZSTS provides access to sales and technical information including production hierarchy
session.findById("wnd[0]").maximize
session.findById("wnd[0]/tbar[0]/okcd").text = "ZSTS"
session.findById("wnd[0]").sendVKey 0

' Verify transaction opened successfully
If Err.Number <> 0 Then
    LogMessage "Error opening ZSTS transaction: " & Err.Description
    WScript.Echo "ERROR: " & Err.Description
    WScript.Quit 1
End If

' Configure search parameters for production hierarchy analysis
' These settings ensure we get the production hierarchy information we need
session.findById("wnd[0]/usr/chkB_PRDHR").selected = true     ' Enable production hierarchy display
' Configure search parameters for production hierarchy analysis
' These settings ensure we get the production hierarchy information we need
session.findById("wnd[0]/usr/chkB_PRDHR").selected = true     ' Enable production hierarchy display
session.findById("wnd[0]/usr/ctxtS_MATNR-LOW").text = materialNumber  ' Enter the material number to search
session.findById("wnd[0]/usr/ctxtS_PLANT-LOW").text = "NL10"           ' Specify plant NL10 for DHL operations
session.findById("wnd[0]/usr/ctxtP_SALES").text = "NL01"               ' Set sales organization for Netherlands
session.findById("wnd[0]/usr/ctxtS_VTWEG-LOW").text = "01"             ' Set distribution channel
session.findById("wnd[0]/usr/chkB_PRDHR").setFocus                     ' Focus on production hierarchy checkbox
session.findById("wnd[0]/tbar[1]/btn[8]").press                        ' Execute the search

' Verify the search executed successfully
If Err.Number <> 0 Then
    LogMessage "Error executing search: " & Err.Description
    WScript.Echo "ERROR: " & Err.Description
    WScript.Quit 1
End If

' Initialize variables for production hierarchy analysis
' These variables will store the results of our search for EUV/TBA indicators
Dim productHierarchy, needsCleaning
needsCleaning = False

' Access the search results grid to extract production hierarchy information
' The grid contains the search results with production hierarchy data
On Error Resume Next
Set grid = session.findById("wnd[0]/usr/cntlGRID1/shellcont/shell")

' Verify grid access was successful
If Err.Number <> 0 Then
    LogMessage "Error accessing grid: " & Err.Description
    WScript.Echo "ERROR: Grid not found - " & Err.Description
    WScript.Quit 1
End If

' Log grid information for debugging purposes
' This helps troubleshoot any issues with data extraction
LogMessage "Grid accessed successfully with " & grid.RowCount & " rows and " & grid.ColumnCount & " columns"

' Analyze grid structure to identify the production hierarchy column
' SAP grids can have varying column names, so we need to detect the correct one
LogMessage "Available columns: "
Dim i, colNames, prodHierarchyColumn
colNames = ""
prodHierarchyColumn = "PRDHA"  ' Default column name for production hierarchy

' Scan all columns to find the production hierarchy field
' We look for column names that contain hierarchy-related keywords
For i = 0 To grid.ColumnCount - 1
    Dim colName
    colName = grid.GetColumnName(i)
    colNames = colNames & colName & " | "
    
    ' Search for common production hierarchy column name patterns
    ' PRDH = Production Hierarchy, HIER = Hierarchy
    If InStr(1, colName, "PRDH", vbTextCompare) > 0 Or _
       InStr(1, colName, "HIER", vbTextCompare) > 0 Then
        prodHierarchyColumn = colName
        LogMessage "Found likely production hierarchy column: " & prodHierarchyColumn
    End If
Next
LogMessage "All columns: " & colNames

' Extract production hierarchy data from the identified column
' This section implements multiple fallback strategies to ensure data retrieval
If grid.RowCount > 0 Then
    ' Primary method: Try to get the value using the identified column name
    On Error Resume Next
    productHierarchy = grid.GetCellValue(0, prodHierarchyColumn)
    
    ' Handle column access errors with alternative approaches
    If Err.Number <> 0 Then
        LogMessage "Error accessing column " & prodHierarchyColumn & ": " & Err.Description
        Err.Clear
        ' Fallback method: Try common alternative column names
        For Each possibleCol In Array("PRDHA", "PRODH", "HIERARCHY", "HIERTX")
            On Error Resume Next
            productHierarchy = grid.GetCellValue(0, possibleCol)
            If Err.Number = 0 Then
                LogMessage "Successfully accessed column: " & possibleCol
                Exit For
            End If
            Err.Clear
        Next
    End If
    
    ' Secondary fallback: Use grid selection method if direct access fails
    ' This approach selects the cell and then retrieves its value
    If productHierarchy = "" Then
        On Error Resume Next
        session.findById("wnd[0]/usr/cntlGRID1/shellcont/shell").currentCellColumn = prodHierarchyColumn
        session.findById("wnd[0]/usr/cntlGRID1/shellcont/shell").selectedRows = "0"
        WScript.Sleep 500
        
        ' Retrieve the selected cell value using the selection method
        productHierarchy = session.findById("wnd[0]/usr/cntlGRID1/shellcont/shell").getCellValue(0, prodHierarchyColumn)
        LogMessage "Retrieved value using selection method: " & productHierarchy
    End If
    
    LogMessage "Found production hierarchy: " & productHierarchy
    
    ' Perform initial EUV/TBA detection for logging purposes
    ' This helps with debugging and provides intermediate feedback
    If InStr(1, productHierarchy, "SPEC EUV", vbTextCompare) > 0 Or _
       InStr(1, productHierarchy, "COM EUV", vbTextCompare) > 0 Or _
       InStr(1, productHierarchy, "TBA", vbTextCompare) > 0 Then
        needsCleaning = True
        LogMessage "Found EUV/TBA indicator in hierarchy: " & productHierarchy
    End If
Else
    ' Handle case where no data rows are found for the material
    LogMessage "No rows found for material"
End If

' Generate final determination about EUV cleaning requirements
' This is the critical business logic that determines cleaning needs based on hierarchy content
' The script looks for specific indicators that identify tools requiring special EUV cleaning:
' - SPEC EUV: Tools used in specialized EUV lithography processes
' - COM EUV: Commercial EUV lithography tools
' - TBA: Tools with "To Be Announced" specifications that may include EUV processes
If InStr(1, productHierarchy, "SPEC EUV", vbTextCompare) > 0 Or _
   InStr(1, productHierarchy, "COM EUV", vbTextCompare) > 0 Or _
   InStr(1, productHierarchy, "TBA", vbTextCompare) > 0 Then
    LogMessage "Found SPEC EUV/COM EUV/TBA in hierarchy: " & productHierarchy
    WScript.Echo "EUV-HIERARCHY:" & productHierarchy
Else
    LogMessage "No SPEC EUV/COM EUV/TBA found in hierarchy: " & productHierarchy
    WScript.Echo "NO-EUV-HIERARCHY:" & productHierarchy
End If

' Clean exit from SAP transaction
' Navigate back to leave the user at a clean starting point
session.findById("wnd[0]").sendVKey 3
session.findById("wnd[0]").sendVKey 3

' Cleanup function to properly close log file and release resources
' This ensures log file integrity and prevents resource leaks
Sub CloseLogFile()
    On Error Resume Next
    If Not logFile Is Nothing Then
        ' Write completion timestamp and separator to log
        LogToFile "Script ended at " & Now
        LogToFile "========================================="
        ' Close file handle and release object reference
        logFile.Close
        Set logFile = Nothing
    End If
End Sub

' Execute final cleanup before script termination
CloseLogFile()